// Copyright (C) 2012-2025 Zammad Foundation, https://zammad-foundation.org/

import { within } from '@testing-library/vue'

import { mockApplicationConfig } from '#tests/support/mock-applicationConfig.ts'

import { mockUserCurrentTwoFactorVerifyMethodConfigurationMutation } from '#shared/entities/user/current/graphql/mutations/two-factor/userCurrentTwoFactorVerifyMethodConfiguration.mocks.ts'
import { mockUserCurrentTwoFactorInitiateMethodConfigurationQuery } from '#shared/entities/user/current/graphql/queries/two-factor/userCurrentTwoFactorInitiateMethodConfiguration.mocks.ts'

import { visitViewAndMockPasswordConfirmation } from '#desktop/pages/personal-setting/__tests__/support/personal-setting-two-factor-auth.ts'

const copyToClipboardMock = vi.fn()

vi.mock('#shared/composables/useCopyToClipboard.ts', async () => ({
  useCopyToClipboard: () => ({ copyToClipboard: copyToClipboardMock }),
}))

describe('Two-factor Authentication - Authenticator App', () => {
  beforeEach(() => {
    mockApplicationConfig({
      two_factor_authentication_method_security_keys: true,
      two_factor_authentication_method_authenticator_app: true,
    })
  })

  it('shows instructions, content and qr-code', async () => {
    const { flyoutContent } = await visitViewAndMockPasswordConfirmation(false, {
      type: 'authenticatorApp',
      configured: false,
      action: 'setup',
    })

    expect(flyoutContent.getByText('Authy'))
    expect(flyoutContent.getByText('Google Authenticator'))
    expect(flyoutContent.getByText('Microsoft Authenticator'))

    const canvas = flyoutContent.getByLabelText('Authenticator app QR code') as HTMLCanvasElement

    expect(canvas.getContext('2d')).not.toBeFalsy()

    expect(
      flyoutContent.getByText('Enter the security code generated by the authenticator app:'),
    ).toBeInTheDocument()
  })

  it('shows secret-code and supports copying to clipboard', async () => {
    const dummySecret = 'AAV7Z6PLEK7'

    mockUserCurrentTwoFactorInitiateMethodConfigurationQuery({
      userCurrentTwoFactorInitiateMethodConfiguration: {
        secret: dummySecret,
      },
    })

    const { view } = await visitViewAndMockPasswordConfirmation(false, {
      type: 'authenticatorApp',
      configured: false,
      action: 'setup',
    })

    await view.events.click(view.getByTestId('secret-overlay'))

    expect(view.getByLabelText('Authenticator app secret')).toHaveTextContent(dummySecret)

    await view.events.click(view.getByRole('button', { name: 'Copy Secret' }))

    expect(copyToClipboardMock).toHaveBeenCalledWith(dummySecret)
  })

  it('supports a11y copying secret code to clipboard', async () => {
    const dummySecret = 'AAV7Z6PLEK7'

    mockUserCurrentTwoFactorInitiateMethodConfigurationQuery({
      userCurrentTwoFactorInitiateMethodConfiguration: {
        secret: dummySecret,
      },
    })

    const { view } = await visitViewAndMockPasswordConfirmation(false, {
      type: 'authenticatorApp',
      configured: false,
      action: 'setup',
    })

    view.getByTestId('secret-overlay').focus()

    await view.events.keyboard('{Enter}')

    expect(view.getByLabelText('Authenticator app secret')).toHaveTextContent(dummySecret)

    await view.events.keyboard('{Tab}{Enter}')

    expect(copyToClipboardMock).toHaveBeenCalledWith(dummySecret)
  })

  it('shows error if security code is wrong', async () => {
    const { view, flyoutContent } = await visitViewAndMockPasswordConfirmation(false, {
      type: 'authenticatorApp',
      configured: false,
      action: 'setup',
    })

    await view.events.type(flyoutContent.getByPlaceholderText('Security Code'), '123456')

    mockUserCurrentTwoFactorVerifyMethodConfigurationMutation({
      userCurrentTwoFactorVerifyMethodConfiguration: {
        errors: [
          {
            field: null,
            message:
              'The verification of the two-factor authentication method configuration failed.',
          },
        ],
        recoveryCodes: null,
      },
    })

    const footer = within(flyoutContent.getByLabelText('Side panel footer'))

    await view.events.click(footer.getByRole('button', { name: 'Set Up' }))

    expect(
      flyoutContent.getByText('Invalid security code! Please try again with a new code.'),
    ).toBeInTheDocument()
  })

  it('shows recovery codes if security code got submitted correctly', async () => {
    const { view, flyout, flyoutContent } = await visitViewAndMockPasswordConfirmation(false, {
      type: 'authenticatorApp',
      configured: false,
      action: 'setup',
    })

    await view.events.type(flyoutContent.getByPlaceholderText('Security Code'), '123456')

    mockUserCurrentTwoFactorVerifyMethodConfigurationMutation({
      userCurrentTwoFactorVerifyMethodConfiguration: {
        errors: null,
        recoveryCodes: ['232'],
      },
    })

    const footer = within(flyoutContent.getByLabelText('Side panel footer'))

    await view.events.click(footer.getByRole('button', { name: 'Set Up' }))

    expect(
      flyoutContent.getByText('Set Up Two-factor Authentication: Save Codes'),
    ).toBeInTheDocument()

    await view.events.click(
      footer.getByRole('button', { name: "OK, I've saved my recovery codes" }),
    )

    expect(flyout).not.toBeInTheDocument()
  })
})
