# Copyright (C) 2012-2025 Zammad Foundation, https://zammad-foundation.org/

class KnowledgeBase::Category::Translation < ApplicationModel
  include HasAgentAllowedParams
  include HasSearchIndexBackend
  include KnowledgeBase::HasUniqueTitle

  AGENT_ALLOWED_ATTRIBUTES = %i[title kb_locale_id].freeze

  belongs_to :kb_locale, class_name: 'KnowledgeBase::Locale', inverse_of: :category_translations

  belongs_to :category,  class_name: 'KnowledgeBase::Category', inverse_of: :translations, touch: true

  validates :title,        presence: true
  validates :kb_locale_id, uniqueness: { case_sensitive: true, scope: :category_id }

  scope :neighbours_of, ->(translation) { joins(:category).where(knowledge_base_categories: { parent_id: translation.category&.parent_id }) }

  def to_param
    [category_id, title.parameterize].join('-')
  end

  def assets(data)
    return data if assets_added_to?(data)

    data = super
    category.assets(data)
  end

  def search_index_attribute_lookup(include_references: true)
    attrs = super

    attrs['title']    = ActionController::Base.helpers.strip_tags attrs['title']
    attrs['scope_id'] = category.parent_id

    attrs
  end

  scope :search_sql_text_fallback, lambda { |query|
    where_or_cis(%w[title], query)
  }

  scope :apply_kb_scope, lambda { |scope|
    if scope.present?
      joins(:category)
        .where(knowledge_base_categories: { parent_id: scope })
    end
  }
end
